/************************************************************************/
/* File: graphPack.c                                                    */
/*                                                                      */
/* This file should be compiled in a project along with the following:  */
/*                                                                      */
/* - XCMDshell.c                                                        */
/* - the MacTraps library                                               */
/*                                                                      */
/* The original code is from 1988.                                      */
/*                                                                      */
/* PRP: March 2025: updated this file for use with THINK C 5, changed   */
/* function definitions from K&R style to ANSI C style, improved        */
/* comments, and made some minor code improvements. Got rid of a number */
/* of NewHandle and NewPtr calls for Str31 strings, as these did not    */
/* seem necessary. Ideally I would use statically-allocated file-scope  */
/* (so-called "global") Str31 variables for temporary strings, instead  */
/* of allocating them on the stack, but when an XCMD is executing, the  */
/* global variable base address in A5 is configured for HyperCard's     */
/* globals, not the XCMD's, and using globals will corrupt HyperCard's  */
/* state. Supposedly there are ways to work around this by saving,      */
/* altering, and restoring A5. I've been able to get this to work in    */
/* other types of MacOS code resources such as INIT files, but not      */
/* XCMDs.                                                               */
/************************************************************************/
#include <HyperXCmd.h> /* declares HyperCard callbacks */
#include <Memory.h>    /* handle routines */
#include <OSEvents.h>  /* declares FlushEvents() */
#include <Resources.h> /* declares ReleaseResource() */
#include <ToolUtils.h> /* declares GetString() */

#include "graphPack.h" /* declares DispatchCommand() */

/*
   myStringID is the ID number of the STR resource (in the HyperCard
   stack hosting this XCMD), which contains the name of the global
   HyperTalk variable (which should be defined in your stack script),
   which will hold a pointer to the window created by the XCMD. This
   allows the same window to persist between calls to the XCMD.
   This sounds overly complicated, but once it is set up, it is simple
   to use from your HyperCard stack.
*/
#define G_NAME_ID 100

/*
   Prototypes for the routines defined in this file.
*/
static short paramToShort(XCmdPtr paramPtr, char *cstr);
static long  paramToLong(XCmdPtr paramPtr, char *cstr);

/*
   Main dispatch function
*/
void SelectCommand(XCmdPtr paramPtr);

/*
   Command handlers
*/
static void makeWindow (XCmdPtr paramPtr);
static void makeLine   (XCmdPtr paramPtr);
static void trashWindow(XCmdPtr paramPtr);
static void cleanWindow(XCmdPtr paramPtr);

/*
    Helper functions to store and rettrieve the window pointer
*/
static void    putWindowPtr(XCmdPtr paramPtr, GrafPtr theWindow,
                            long strId);
static GrafPtr getWindowPtr(XCmdPtr paramPtr, long strID);

/************************************************************************/
/* A helper function to turn an incoming parameter into a short. This   */
/* uses glue functions to make two callbacks into HyperCard. The        */
/* parameters come in as zero-terminated C-style strings. Convert the   */
/* parameter to a Pascal-style string with a length byte, then turn it  */
/* into a short.                                                        */
/************************************************************************/
static short paramToShort(XCmdPtr paramPtr, char *cstr)
{
    Str31 param_buf;

    ZeroToPas(paramPtr, cstr, (StringPtr)param_buf);
    return (short)StrToNum(paramPtr, param_buf);
}

/************************************************************************/
/* A similar helper function that returns a long.                       */
/************************************************************************/
static long paramToLong(XCmdPtr paramPtr, char *cstr)
{
    Str31 param_buf;

    ZeroToPas(paramPtr, cstr, (StringPtr)param_buf);
    return (long)StrToNum(paramPtr, param_buf);
}

/************************************************************************/
/* This is the dispatcher which determines which one of the routines    */
/* in the graphPack XCMD to run.                                        */
/************************************************************************/
void SelectCommand(XCmdPtr paramPtr)
{
    short   which_command;
    GrafPtr saved_port;

    /* Save HyperCard's GrafPort */
    GetPort(&saved_port);

    PurgeMem((Size)5000);

    which_command = paramToShort(paramPtr, (char*)*paramPtr->params[0]);

    switch (which_command)
    {
        case 1:
            if (paramPtr->paramCount == 2)
            {
                makeWindow(paramPtr);
            }
            else
            {
                SysBeep(40);
            }
            break;
 
        case 2:
            if (paramPtr->paramCount == 5)
            {
                makeLine(paramPtr);
            }
            else
            {
                SysBeep(40);
            }
            break;

        case 3:
            if (paramPtr->paramCount == 2)
            {
                cleanWindow(paramPtr);
            }
            else
            {
                SysBeep(40);
            }
            break;

        case 9:
            if (paramPtr->paramCount == 1)
            {
                trashWindow(paramPtr);
            }
            else
            {
                SysBeep(40);
            }
            break;
    }
    FlushEvents (everyEvent, 0);

    /* Restore HyperCard's GrafPort */
    SetPort(saved_port);
}

/************************************************************************/
/* This function will initialize a dBoxProc window at the location of   */
/* the PICT read, and draw the PICT in it. To start with a blank window */
/* use an empty PICT                                                    */
/************************************************************************/
static void makeWindow (XCmdPtr paramPtr)
{
    long      which_pict;
    Rect      bounding_rect;
    WindowPtr theWindow;
    PicHandle myWindowBox;

    which_pict = paramToLong(paramPtr, (char*)*paramPtr->params[1]);

    myWindowBox = GetPicture(which_pict);
    bounding_rect = (*myWindowBox)->picFrame;

    theWindow = NewWindow(0L, &bounding_rect, "\P", true,
                          dBoxProc, (WindowPtr)-1L, false, 0L);
    SetPort(theWindow);
    DrawPicture(myWindowBox, &theWindow->portRect);
    ReleaseResource((Handle)myWindowBox);

    putWindowPtr(paramPtr, theWindow, G_NAME_ID);
}

/************************************************************************/
/* This function reads two coordinate pairs in order to draw a line     */
/* segment in a window owned by the XCMD.                               */
/************************************************************************/
void makeLine(XCmdPtr paramPtr)
{
    GrafPtr xcmdWindow;
    short   horiz, vert, newhoriz, newvert;
 
    horiz    = paramToShort(paramPtr, (char*)*paramPtr->params[1]);
    vert     = paramToShort(paramPtr, (char*)*paramPtr->params[2]);
    newhoriz = paramToShort(paramPtr, (char*)*paramPtr->params[3]);
    newvert  = paramToShort(paramPtr, (char*)*paramPtr->params[4]);

    xcmdWindow = getWindowPtr(paramPtr, G_NAME_ID);
    SetPort(xcmdWindow);
    ForeColor (magentaColor);
    PenSize(2,2);
    MoveTo (horiz, vert);
    LineTo (newhoriz, newvert);
}

/************************************************************************/
/* This function will redraw the window owned by the XCMD. This will    */
/* effectively erase anything drawn in the window since it was created. */
/* The first parameter is the ID of the PICT resource to be opened.     */
/* This can be the PICT resource that was used with createWindow, or a  */
/* different one.                                                       */
/************************************************************************/
void cleanWindow(XCmdPtr paramPtr)
{
    GrafPtr   xcmdWindow;
    long      which_pict;
    PicHandle myPict;

    which_pict = paramToLong(paramPtr, (char*)*paramPtr->params[1]);

    myPict = GetPicture(which_pict);

    /* Retrieve the window pointer from the HyperTalk global */
    xcmdWindow = getWindowPtr(paramPtr, G_NAME_ID);

    /* Draw the PICT */
    SetPort(xcmdWindow);
    DrawPicture(myPict, &xcmdWindow->portRect);
    ReleaseResource((Handle)myPict);
}

/************************************************************************/
/* This function will get the current window pointer from HyperCard,    */
/* set the global variable that holds it to NULL, and dispose of the    */
/* window.                                                              */
/************************************************************************/
void trashWindow(XCmdPtr paramPtr)
{
    GrafPtr xcmdWindow = getWindowPtr(paramPtr, G_NAME_ID);

    DisposeWindow(xcmdWindow);

    /* Store NULL into the HyperTalk global */
    putWindowPtr(paramPtr, (GrafPtr)0, G_NAME_ID);
}

/************************************************************************/
/* This routine will call back into Hypercard to store a pointer to the */
/* window, in string form. The strID is used to look up  the string     */
/* resource, which should be in the HyperCard stack, containing the     */
/* name of the HyperTalk global variable to store the pointer in.       */
/************************************************************************/
void putWindowPtr(XCmdPtr paramPtr, GrafPtr theWindow, long strID)
{
    Str31        param_buf;
    StringHandle global_name;
    Handle       new_value;

    /* Put the GrafPtr, in Pascal string form, into the given buffer. */
    LongToStr(paramPtr, (long)theWindow, param_buf);

    /* Get the global variable name as a handle to a string resource. */
    global_name = GetString(strID);

    /* Get the new value as a handle to a C string. */
    new_value = PasToZero(paramPtr, (StringPtr)param_buf);

    /* Set the global variable to the new value. */
    SetGlobal(paramPtr, (StringPtr)*global_name, (Handle)new_value);

    ReleaseResource((Handle)global_name);
    DisposHandle(new_value);
}

/************************************************************************/
/* This function will return a pointer to the window owned by the       */
/* graphPack XCMD graphPack. It accepts the ID number of the string     */
/* resource which contains the name of the HyperTalk variable holding   */
/* the pointer.                                                         */
GrafPtr getWindowPtr(XCmdPtr paramPtr, long strID)
{
    GrafPtr      xCmdWindow;
    StringHandle global_name;
    Handle       value;

    /* Get the global variable name as a handle to a string resource. */
    global_name = GetString(strID);

    /* Get the value of the HyperCard global as a handle to a C string */
    value = GetGlobal(paramPtr, (StringPtr)*global_name);

    xCmdWindow = (GrafPtr)paramToLong(paramPtr, (char *)*value);

    ReleaseResource((Handle)global_name);
    DisposHandle(value);

    return xCmdWindow;
}
